<?php

namespace App\Models;

use App\Models\Group;
use App\Models\ScheduledTaskResource;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Resource extends Model
{
    use HasFactory;
    use SoftDeletes;

    public const CHART_FULL_SIZE    = 1;
    public const CHART_HALF_SIZE    = 2;
    public const CHART_QUARTER_SIZE = 3;

    protected $primaryKey = 'id';
    protected $connection = 'mysql';

    public $fillable = [
        'name',
        'url',
        'access_control_type',
        'module_id',
        'category_id',
        'resource_creator',
        'data_connection_id',
        'metric_color',
        'info_box_icon',
        'widget_size'
    ];

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);
        $prefix      = config('srm_config.installer.table_prefix', 'srm_');
        $this->table = "{$prefix}analytics_resources";
    }

    public static function tableName()
    {
        return with(new static())->getTable();
    }

    public function module()
    {
        return $this->belongsTo(Module::class, 'module_id');
    }

    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }

    public function data_connection()
    {
        return $this->belongsTo(DataConnection::class, 'data_connection_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'resource_creator');
    }

    public function scheduledTaskResource()
    {
        return $this->hasMany(ScheduledTaskResource::class, 'resource_id');
    }

    public function configurations()
    {
        return $this->hasOne(ResourceConfiguration::class, 'Resource_id');
    }

    public function homeWidgets()
    {
        return $this->hasMany(HomeDashboard::class, 'resource_id');
    }

    public function dashboardWidgets()
    {
        return $this->hasMany(DashboardWidget::class, 'resource_id');
    }

    public function resource_permissions()
    {
        $prefix = config("srm_config.installer.table_prefix", "srm_");

        return $this->belongsToMany(Group::class, $prefix . 'resource_permissions', 'resource_id', 'group_id')
            ->withPivot('group_id');
    }

    public function getGroupsCanAccessResource()
    {

        if (isset($this->resource_permissions)) {
            return $this->resource_permissions->pluck("name");
        }

    }

    public function getReportUrl()
    {
        return getBaseUrl() . getReportPath($this->url);
    }

    public function getResourceType()
    {
        return $this->module->name;
    }

    public function getReportPdf()
    {
        $reportPath = explode('/', $this->getReportUrl());

        array_pop($reportPath);

        $reportPath = implode('/', $reportPath);

        return request()->server('DOCUMENT_ROOT') . "$reportPath" . "/" . str_replace(' ', '', $this->name) . '-singlepage.pdf';
    }

    public function isChartFullSize()
    {
        return $this->widget_size == Resource::CHART_FULL_SIZE;
    }

    public function isChartHalfSize()
    {
        return $this->widget_size === Resource::CHART_HALF_SIZE;
    }

    public function getResourceConfiguration($key)
    {
        return $this->configurations->Json_configurations[$key] ?? "";
    }

    public function getConnectionAsString()
    {
        return isset($this->data_connection_id) ? $this->data_connection_id : "default";
    }

    public static function findResourceByName($chart)
    {
        return static::where('name', $chart);
    }

    public function getChartDateRangeTimeScale()
    {

        if (!is_null($this->configurations->chart_type)) {

            switch ($this->getResourceConfiguration('date_range')) {
                case 'last_5_years':
                    return ["years" => 'Years'];
                    break;
                case 'this_year':
                case 'last_12_months':
                    return [
                        "quarters" => "Quarters",
                        "months"   => "Months"
                    ];
                    break;
                case 'this_6_months':
                case 'last_6_months':
                case 'this_3_months':
                case 'last_3_months':
                    return ["months" => "Months"];
                    break;
                case 'this_month':
                case 'last_30_days':
                    return [
                        "days"  => "Days",
                        "weeks" => "Weeks"
                    ];
                    break;
                case 'this_week':
                case 'last_7_days':
                    return [
                        "days" => "Days"
                    ];
                    break;
                case 'today':
                case 'yesterday':
                    return [
                        "hours" => "Hours"
                    ];
                    break;
                default:
                    return ["years" => 'Years'];
                    break;
            }

        }

    }

    public function getMetricBgColor()
    {

        switch ($this->metric_color) {
            case 'blue':
                return 'bg-info';
                break;
            case 'green':
                return 'bg-success';
                break;
            case 'orange':
                return 'bg-warning';
                break;
            case 'red':
                return 'bg-danger';
                break;
            case 'white':
                return 'bg-white';
                break;
            case 'purple':
                return 'bg-purple';
                break;
            default:
                return "";
                break;
        }

    }

    public function getMetricBgColorHexaDecimal()
    {

        switch ($this->metric_color) {
            case 'blue':
                return '#17a2b8';
                break;
            case 'green':
                return '#28a745';
                break;
            case 'orange':
                return '#ffc107';
                break;
            case 'red':
                return '#dc3545';
                break;
            case 'white':
                return '#1f2d3d';
                break;
            case 'purple':
                return '#a020f0';
                break;
            default:
                return "";
                break;
        }

    }

    public function getResourceIcon()
    {
        return "&" . $this->info_box_icon;
    }

}
